const API_KEY = ''; // ←YouTube Data API v3のAPIキーをここに入力

module.exports = {
  name: 'ハンドル名対応(チャンネル名表示)プラグイン',
  uid: 'tokyo.nvl-game.youtube-handle-to-name',
  version: '0.1.8-beta',
  author: '深紅の滴',
  url: 'https://nvl-game.tokyo/plugins/5578',
  permissions: ['filter.comment'],
  defaultState: {},

  // キャッシュリセット処理を共通化
  resetCaches() {
    this.channelNameCache = {};
    this.channelNamePromiseCache = {};
    this.senderNameCache = {};
    this.senderPromiseCache = {};
  },

  init({ dir, store }) {
    this.store = store;
    this.resetCaches();

    if (this.ptMidnightTimer) clearTimeout(this.ptMidnightTimer);
    const setNextPTMidnight = () => {
      const now = new Date();
      const utc = now.getTime() + now.getTimezoneOffset() * 60000;
      const jan = new Date(now.getFullYear(), 0, 1).getTimezoneOffset();
      const jul = new Date(now.getFullYear(), 6, 1).getTimezoneOffset();
      const dstOffset = Math.max(jan, jul);
      const ptNow = new Date(utc - dstOffset * 60000);
      const ptMidnight = new Date(ptNow);
      ptMidnight.setHours(24, 0, 0, 0);
      const msUntilMidnight = ptMidnight - ptNow;
      this.ptMidnightTimer = setTimeout(() => {
        this.resetCaches();
        setNextPTMidnight();
        console.log('[YouTubeNamePlugin] PT午前0時にキャッシュを削除しました');
      }, msUntilMidnight);
    };
    setNextPTMidnight();
  },

  destroy() {
    this.resetCaches();
    if (this.ptMidnightTimer) {
      clearTimeout(this.ptMidnightTimer);
      this.ptMidnightTimer = null;
    }
  },

  // API呼び出しを共通化
  async fetchYouTubeAPI(url, errorContext) {
    try {
      const res = await fetch(url);
      if (!res.ok) throw new Error(`API error: ${res.status}`);
      return await res.json();
    } catch (e) {
      console.error(`[YouTubeNamePlugin] ${errorContext}:`, e);
      return null;
    }
  },

  async filterComment(comment, service, userData) {
    if (comment.service !== 'youtube' || !comment.data?.userId || !comment.data?.name) {
      return comment;
    }

    const fetchSenderNameByHandle = async (handle) => {
      if (this.senderNameCache[handle]) return this.senderNameCache[handle];
      if (this.senderPromiseCache[handle]) return await this.senderPromiseCache[handle];

      const promise = (async () => {
        if (!API_KEY) return handle;
        const url = `https://www.googleapis.com/youtube/v3/channels?part=snippet&forHandle=${handle}&key=${API_KEY}`;
        const data = await this.fetchYouTubeAPI(url, `Error fetching sender by handle @${handle}`);

        if (data?.items?.length > 0) {
          const senderName = data.items[0].snippet.title;
          this.senderNameCache[handle] = senderName;
          return senderName;
        }
        // APIで取得できなかった場合はハンドル名（@なし）を返す
        this.senderNameCache[handle] = handle;
        return handle;
      })();

      this.senderPromiseCache[handle] = promise;
      try {
        return await promise;
      } finally {
        delete this.senderPromiseCache[handle];
      }
    };

    const userId = comment.data.userId;
    const originalCommentName = comment.data.name;
    const channelId = userId.replace(/^yt-/, '');

    // 表示ロジックを刷新
    const applyNameChange = async (comment, cache, originalName) => {
      const apiName = cache.name; // APIで取得したユーザー名
      const apiHandle = cache.handle; // APIで取得したハンドル名（@なし）
      const correctScreenName = `@${apiHandle}`; // APIで取得した正しいハンドル名（@付き）

      if (originalName.startsWith('@')) {
        // ケースB: コメントの名前が@で始まる場合
        if (originalName === apiName) {
          // 「@で始まるユーザー名」でのコメントだった場合
          comment.data.displayName = apiName;
          comment.data.name = apiName;
          comment.data.screenName = correctScreenName; // ハンドル名を追加表示
        } else {
          // 「ハンドル名」でのコメントだった場合
          comment.data.displayName = apiName; // 表示名をユーザー名に置き換え
          comment.data.name = apiName;
          comment.data.screenName = originalName; // 元のハンドル名を表示
        }
      } else {
        // ケースA: コメントの名前が@で始まらない場合（ユーザー名でのコメント）
        comment.data.displayName = apiName;
        comment.data.name = apiName;
        comment.data.screenName = correctScreenName; // ハンドル名を追加表示
      }

      // ギフトコメントの置換
      if (comment.data.isSponsorshipGiftReceiver && comment.data.comment) {
        // 送信者名抽出（ハンドル or 既存名）
        const giftSenderMatch = comment.data.comment.match(/さんが ([^ ]+) さんから/);
        if (giftSenderMatch?.[1]) {
          const senderHandleOrName = giftSenderMatch[1].replace(/^@/, '');
          // 送信者名を取得
          let senderName = await fetchSenderNameByHandle(senderHandleOrName);
          comment.data.comment = `${comment.data.displayName}さんが ${senderName} さんからメンバーシップ ギフトを受け取りました`;
        }
      }

      // speechText内の@xxxxをユーザー名またはニックネームに置換
      if (comment.data.speechText) {
        const handles = [...comment.data.speechText.matchAll(/@([^\s　]+)/g)];
        let replacedText = comment.data.speechText;
        for (const match of handles) {
          const handle = match[1];
          let name = handle;

          // userDataからnickNameやnameを参照
          let userInfo = null;
          if (userData && typeof userData === 'object') {
            userInfo = Object.values(userData).find(u =>
              u.screenName && u.screenName.replace(/^@/, '').toLowerCase() === handle.toLowerCase()
            );
          }

          if (userInfo && userInfo.nickName) {
            name = userInfo.nickName;
          } else if (userInfo && userInfo.name) {
            name = userInfo.name;
          } else {
            name = await fetchSenderNameByHandle(handle);
          }

          replacedText = replacedText.replaceAll(`@${handle}`, name);
        }
        comment.data.speechText = replacedText;
      }

      return comment;
    };

    // チャンネル情報取得処理
    const getChannelInfo = async (id) => {
      if (this.channelNameCache[id]) return this.channelNameCache[id];
      if (this.channelNamePromiseCache[id]) return await this.channelNamePromiseCache[id];
      if (!API_KEY) return null;

      const promise = (async () => {
        // partにbrandingSettingsを追加して、より正確なハンドル名を取得
        const url = `https://www.googleapis.com/youtube/v3/channels?part=snippet,brandingSettings&id=${id}&key=${API_KEY}`;
        const data = await this.fetchYouTubeAPI(url, `Error fetching channel info for ${id}`);
        if (data?.items?.length > 0) {
          const item = data.items[0];
          const channelName = item.snippet.title;
          let handle = item.snippet.customUrl; // フォールバック用のハンドル

          // 優先: brandingSettings.channel.keywords からハンドルを抽出
          const keywords = item.brandingSettings?.channel?.keywords;
          if (keywords) {
            // keywordsはスペース区切りの文字列 "keyword1" "keyword2" "@Handle"
            const match = keywords.match(/"(@[^"]+)"/);
            if (match?.[1]) {
              handle = match[1]; // @付きのハンドル名が取得できる
            }
          }

          if (channelName && handle) {
            // @を削除してキャッシュに保存
            const handleWithoutAt = handle.startsWith('@') ? handle.substring(1) : handle;
            this.channelNameCache[id] = { name: channelName, handle: handleWithoutAt, fetchedAt: Date.now() };
            return this.channelNameCache[id];
          }
        }
        return null;
      })();

      this.channelNamePromiseCache[id] = promise;
      try {
        return await promise;
      } finally {
        delete this.channelNamePromiseCache[id];
      }
    };

    const channelInfo = await getChannelInfo(channelId);
    if (channelInfo) {
      return await applyNameChange(comment, channelInfo, originalCommentName);
    }

    return comment;
  },
};