const app = Vue.createApp({
  setup() {
    document.body.removeAttribute('hidden')
  },
  data () {
    return {
      pollStart: null,
      pollUpdate: null,
      pollFinish: null,
      loading: true // 初期ロード状態
    }
  },
  methods: {
    getPollPercent(count) {
      const poll = this.displayPoll
      if (!poll || !poll.total) return 0;
      return ((count / poll.total) * 100).toFixed(1);
    },
    /**
     * imgタグを「src画像 > alt絵文字 > どちらもなければ除去」の優先順位ですべて処理
     * @param {string} html
     * @returns {Array<{type: 'image'|'text', value: string}>}
     */
    extractEmojis(html) {
      if (!html) return [];
      // すべてのimgタグを抽出し、間のテキストも含めて配列化
      const imgTagRegex = /<img([\s\S]*?)\/?>/gi;
      let result = [];
      let lastIndex = 0;
      let match;
      while ((match = imgTagRegex.exec(html)) !== null) {
        // imgタグ前のテキストも追加
        if (match.index > lastIndex) {
          const text = html.slice(lastIndex, match.index);
          if (text) result.push({ type: 'text', value: text });
        }
        const attrs = match[1];
        const srcMatch = attrs.match(/\ssrc\s*=\s*(['"])(.*?)\1/i);
        const altMatch = attrs.match(/\salt\s*=\s*(['"])(.*?)\1/i);
        const srcVal = srcMatch && srcMatch[2] && srcMatch[2].trim() !== '' ? srcMatch[2].trim() : '';
        const altVal = altMatch && altMatch[2] && altMatch[2].trim() !== '' ? altMatch[2].trim() : '';
        if (srcVal) {
          result.push({ type: 'image', value: srcVal });
        } else if (altVal) {
          result.push({ type: 'text', value: altVal });
        }
        // どちらもなければ何も追加しない（除去）
        lastIndex = imgTagRegex.lastIndex;
      }
      // imgタグの後ろのテキストも追加
      if (lastIndex < html.length) {
        const text = html.slice(lastIndex);
        if (text) result.push({ type: 'text', value: text });
      }
      return result;
    }
  },
  mounted () {
    OneSDK.setup({
      permissions: OneSDK.usePermission([OneSDK.PERM.YT_SURVEY]),
    })
    // metaから合計票数を抽出
    const extractTotal = (meta) => {
      if (!meta) return 0;
      const match = meta.match(/([0-9,]+)\s*票/);
      if (match) {
        return parseInt(match[1].replace(/,/g, ''), 10);
      }
      return 0;
    };
    const convertPoll = (poll) => {
      if (!poll) return null;
      const total = extractTotal(poll.meta);
      return {
        // titleにもextractEmojisを適用
        title: this.extractEmojis(poll.question || poll.title || ''),
        options: (poll.choices || poll.options || []).map(opt => {
          // ratioから票数を計算
          const count = typeof opt.ratio === 'number' && total
            ? Math.round(opt.ratio * total)
            : 0;
          return {
            ...opt,
            // textからimgタグを「src画像 > alt絵文字 > どちらもなければ除去」で全て処理
            text: this.extractEmojis(opt.text || opt.label || ''),
            count
          };
        }),
        total
      };
    };
    // yt.survey.start
    OneSDK.subscribe({
      action: 'yt.survey.start',
      callback: (poll) => {
        if (this.pollUpdate) return
        this.pollStart = convertPoll.call(this, poll)
        this.pollFinish = null
        this.loading = false
      }
    })
    // yt.survey.update
    OneSDK.subscribe({
      action: 'yt.survey.update',
      callback: (poll) => {
        this.pollUpdate = convertPoll.call(this, poll)
        this.pollStart = null
        this.pollFinish = null
        this.loading = false
      }
    })
    // yt.survey.finish
    OneSDK.subscribe({
      action: 'yt.survey.finish',
      callback: (poll) => {
        this.pollFinish = convertPoll.call(this, poll)
        this.pollUpdate = null
        this.pollStart = null
        this.loading = false
      }
    })
    OneSDK.connect()
    // タイムアウト処理は削除。アンケートが始まるまでloadingはtrueのまま
  },
  computed: {
    displayPoll() {
      if (this.pollUpdate) return this.pollUpdate
      if (this.pollFinish) return this.pollFinish
      if (this.pollStart)  return this.pollStart
      return null
    }
  }
})
OneSDK.ready().then(() => {
  app.mount("#container");
})